subroutine energyandforces_periodic(nat,alat,rxyz,epot,fxyz,deralat)
 !energy and its derivatives for the model potential: .02*(-4*((x/5.)-1)**4+6*((x/5.)-1)**6)/(x/5.)**4 with periodic boundary conditions
 !input: nat: number of atoms
 !       alat: the 3 lattice vectors specifying the cell arranged in the colums of alat
 !       rxyz: positions of atoms
 !output: epot: potential energy
 !        fxyz: forces (negative derivative of energy with respect to positions
 !        deralat: negative derivative of energy with respect to the lattice vectors
 IMPLICIT none
 !input and output variables
 INTEGER :: nat !number of atoms
 REAL(8), DIMENSION(3,3) :: alat !lattice vectors
 REAL(8), DIMENSION(3,nat) :: rxyz !atom positions
 REAL(8) :: epot !potential energy
 REAL(8), DIMENSION(3,nat) :: fxyz !forces
 REAL(8), DIMENSION(3,3) :: deralat !derivative of the lattice vecotrs

 !local variables
 INTEGER :: i
 INTEGER :: j
 INTEGER :: l
 INTEGER :: iat
 INTEGER :: jat
 INTEGER :: jx
 INTEGER :: jy
 INTEGER :: jz
 INTEGER :: nxyzmax
 REAL(8) :: t1
 REAL(8) :: t2
 REAL(8) :: t3
 REAL(8) :: tt
 REAL(8) :: xj
 REAL(8) :: yj
 REAL(8) :: zj
 REAL(8) :: dd
 REAL(8) :: dd2
 REAL(8) :: dd4
 REAL(8) :: dd4i
 REAL(8) :: dd5i
 REAL(8) :: ddsq
 REAL(8) :: ddsq2
 REAL(8) :: ddsqi
 REAL(8) :: de
 REAL(8) :: dx
 REAL(8) :: dy
 REAL(8) :: dz
 REAL(8), DIMENSION(3,3) :: alatalat
 REAL(8), DIMENSION(3) :: eigalat !eigenvalues of derivative lattice vecotrs
 REAL(8), DIMENSION(3,nat) :: xyzred !reduced coordinates


 ! make sure all atoms are in the cell
 CALL cart2frac(nat,alat,rxyz,xyzred)
 DO iat=1,nat
   DO l=1,3
     xyzred(l,iat)=modulo(xyzred(l,iat),1.d0)
   ENDDO
  ENDDO
  CALL frac2cart(nat,alat,xyzred,rxyz)

  ! nxyzmax specified over how many neighboring cells one has to search to account for all atoms within the interaction range of 5
  DO i=1,3
    DO j=1,3
      alatalat(i,j)=alat(1,i)*alat(1,j)+alat(2,i)*alat(2,j)+alat(3,i)*alat(3,j)
    ENDDO
  ENDDO
  CALL eigenvalues(alatalat,eigalat)
  nxyzmax= int(sqrt(1.d0/eigalat(1))*5.d0) + 1

  epot=0.d0
  DO iat=1,nat
    fxyz(1,iat)=0.d0
    fxyz(2,iat)=0.d0
    fxyz(3,iat)=0.d0
  ENDDO
  DO i=1,3
    DO j=1,3
      deralat(i,j)=0.d0
    ENDDO
  ENDDO


  DO iat = 1, nat  ! loop over all atoms in the cell
    DO jat = 1, nat  ! loop over all atom within the range of the potential
      DO jx = -nxyzmax,nxyzmax
        DO jy = -nxyzmax,nxyzmax
          DO jz = -nxyzmax,nxyzmax
            xj = rxyz(1, jat) + jx*alat(1,1)+jy*alat(1,2)+jz*alat(1,3)
            yj = rxyz(2, jat) + jx*alat(2,1)+jy*alat(2,2)+jz*alat(2,3)
            zj = rxyz(3, jat) + jx*alat(3,1)+jy*alat(3,2)+jz*alat(3,3)
            dx=rxyz(1,iat)-xj
            dy=rxyz(2,iat)-yj
            dz=rxyz(3,iat)-zj
            ddsq2=dx**2+dy**2+dz**2
            IF (ddsq2.lt.1.d-4 .and. iat.ne.jat) THEN
              WRITE(*,*) 'WARNING: atoms too close, presumably time step too large ',iat,jat,jx,jy,jz,ddsq2
              STOP
            ENDIF
            IF (ddsq2.gt.1.d-10 .and. ddsq2.lt.25.d0) THEN
              ddsq=sqrt(ddsq2)
              ddsqi=1.d0/ddsq
              dd=.2d0*ddsq
              dd2=.04d0*ddsq2
              dd4=dd2*dd2
              dd4i=1.d0/dd4
              dd5i=1.d0/(dd4*dd)

              de=.02d0*( 6.d0*(dd-1.d0)**6 - 4.d0*(dd-1.d0)**4 )
              epot=epot+de*dd4i
              tt=0.004d0*(36.d0*(dd-1.d0)**5 - 16.d0*(dd-1.d0)**3)*dd4i-0.8d0*de*dd5i
              t1=dx*tt*ddsqi
              t2=dy*tt*ddsqi
              t3=dz*tt*ddsqi
              fxyz(1,iat)=fxyz(1,iat)-2.d0*t1
              fxyz(2,iat)=fxyz(2,iat)-2.d0*t2
              fxyz(3,iat)=fxyz(3,iat)-2.d0*t3

              deralat(1,1)=deralat(1,1)-t1*(xyzred(1,iat)-xyzred(1,jat)-jx)
              deralat(1,2)=deralat(1,2)-t1*(xyzred(2,iat)-xyzred(2,jat)-jy)
              deralat(1,3)=deralat(1,3)-t1*(xyzred(3,iat)-xyzred(3,jat)-jz)

              deralat(2,1)=deralat(2,1)-t2*(xyzred(1,iat)-xyzred(1,jat)-jx)
              deralat(2,2)=deralat(2,2)-t2*(xyzred(2,iat)-xyzred(2,jat)-jy)
              deralat(2,3)=deralat(2,3)-t2*(xyzred(3,iat)-xyzred(3,jat)-jz)

              deralat(3,1)=deralat(3,1)-t3*(xyzred(1,iat)-xyzred(1,jat)-jx)
              deralat(3,2)=deralat(3,2)-t3*(xyzred(2,iat)-xyzred(2,jat)-jy)
              deralat(3,3)=deralat(3,3)-t3*(xyzred(3,iat)-xyzred(3,jat)-jz)
            ENDIF
          ENDDO
        ENDDO
      ENDDO
    ENDDO
  ENDDO

end subroutine energyandforces_periodic


subroutine eigenvalues(alatalat, eigalat)
  IMPLICIT none
  !i/o variables
  REAL(8), DIMENSION(3,3) :: alatalat
  REAL(8), DIMENSION(3) :: eigalat
  !local variables
  REAL(8), DIMENSION(3,3) :: B
  REAL(8), DIMENSION(3,3) :: I
  REAL(8), PARAMETER :: pi=3.14
  REAL(8) :: p1
  REAL(8) :: q
  REAL(8) :: p2
  REAL(8) :: p
  REAL(8) :: detB
  REAL(8) :: r
  REAL(8) :: phi

  I(:,:) = 0.d0
  I(1,1) = 1.d0
  I(2,2) = 1.d0
  I(3,3) = 1.d0

  p1 = alatalat(1,2)*alatalat(1,2) + alatalat(1,3)*alatalat(1,3) + alatalat(2,3)*alatalat(2,3)
  IF (p1 .eq. 0.d0) THEN
    eigalat(1) = alatalat(1,1)
    eigalat(2) = alatalat(2,2)
    eigalat(3) = alatalat(3,3)
  ELSE
    q = alatalat(1,1)+alatalat(2,2)+alatalat(3,3)
    q = q/3.d0
    p2 = (alatalat(1,1)-q)*(alatalat(1,1)-q) + (alatalat(2,2)-q)*(alatalat(2,2)-q) +&
         (alatalat(3,3)-q)*(alatalat(3,3)-q) + 2.d0 * p1
    p = sqrt(p2/6.d0)
    B = (1.d0/p) * (alatalat - q*I)
    detB =  B(1,1)*B(2,2)*B(3,3)  &
          - B(1,1)*B(2,3)*B(3,2)  &
          - B(1,2)*B(2,1)*B(3,3)  &
          + B(1,2)*B(2,3)*B(3,1)  &
          + B(1,3)*B(2,1)*B(3,2)  &
          - B(1,3)*B(2,2)*B(3,1)
    r = detB / 2.d0

    IF ( r .le. -1.d0) THEN
      phi = pi/3.d0
    ELSEIF ( r .ge. 1.d0) THEN
      phi = 0
    ELSE
      phi = acos(r) / 3.d0
    ENDIF

    eigalat(1) = q + 2.d0 * p * cos(phi)
    eigalat(3) = q + 2.d0 * p * cos(phi + (2.d0*pi/3.d0))
    eigalat(2) = 3.d0 * q - eigalat(1) - eigalat(3)

  ENDIF
end subroutine eigenvalues



     subroutine frac2cart(nat, alat, xyzred, rxyz)
  !converts reduced coordinates xyzred to cartesian coordinates rxyz
     IMPLICIT NONE
     REAL(8) :: alat(3,3)
     REAL(8) :: rxyzred(3,nat)
     REAL(8) :: rxyz(3,nat)
     REAL(8) :: t
     INTEGER :: iat
     INTEGER :: i, j

     DO iat=1,nat
        DO i = 1, 3
           t = 0.d0
           DO j = 1, 3
              t = t + xyzred(j,iat) * alat(i, j)
           END DO
           rxyz(i,iat) = t
        END DO
     END DO

    !  do j=1,3
    !  do i=1,3
    !  alat(i,j)=alat(i,j)
    !  enddo
    !  enddo

     end subroutine frac2cart




subroutine cart2frac(nat,alat,rxyz,xyzred)
  !converts cartesian coordinates rxyz to reduced coordinates xyzred
  IMPLICIT none
  !i/o variables
  INTEGER :: nat
  REAL(8), DIMENSION(3,3) :: alat
  REAL(8), DIMENSION(3,nat) :: rxyz
  REAL(8), DIMENSION(3,nat) :: xyzred

  !local variables
  INTEGER :: iat
  REAL(8) :: div
  REAL(8), DIMENSION(3,3) :: alatinv

  div=alat(1,1)*alat(2,2)*alat(3,3)-alat(1,1)*alat(2,3)*alat(3,2)- &
      alat(1,2)*alat(2,1)*alat(3,3)+alat(1,2)*alat(2,3)*alat(3,1)+ &
      alat(1,3)*alat(2,1)*alat(3,2)-alat(1,3)*alat(2,2)*alat(3,1)
  div=1.d0/div
  alatinv(1,1) = (alat(2,2)*alat(3,3)-alat(2,3)*alat(3,2))*div
  alatinv(1,2) =-(alat(1,2)*alat(3,3)-alat(1,3)*alat(3,2))*div
  alatinv(1,3) = (alat(1,2)*alat(2,3)-alat(1,3)*alat(2,2))*div
  alatinv(2,1) =-(alat(2,1)*alat(3,3)-alat(2,3)*alat(3,1))*div
  alatinv(2,2) = (alat(1,1)*alat(3,3)-alat(1,3)*alat(3,1))*div
  alatinv(2,3) =-(alat(1,1)*alat(2,3)-alat(1,3)*alat(2,1))*div
  alatinv(3,1) = (alat(2,1)*alat(3,2)-alat(2,2)*alat(3,1))*div
  alatinv(3,2) =-(alat(1,1)*alat(3,2)-alat(1,2)*alat(3,1))*div
  alatinv(3,3) = (alat(1,1)*alat(2,2)-alat(1,2)*alat(2,1))*div

  DO iat=1,nat
    xyzred(1,iat)=alatinv(1,1)*rxyz(1,iat)+alatinv(1,2)*rxyz(2,iat)+alatinv(1,3)*rxyz(3,iat)
    xyzred(2,iat)=alatinv(2,1)*rxyz(1,iat)+alatinv(2,2)*rxyz(2,iat)+alatinv(2,3)*rxyz(3,iat)
    xyzred(3,iat)=alatinv(3,1)*rxyz(1,iat)+alatinv(3,2)*rxyz(2,iat)+alatinv(3,3)*rxyz(3,iat)
  ENDDO

 end subroutine



subroutine back2cell(nat,rxyz,alat)
!    translates atoms outside the cell back into the cell
  IMPLICIT none
  !i/o variables
  INTEGER :: nat
  REAL(8), DIMENSION(3,nat) :: rxyz
  REAL(8), DIMENSION(3,3) :: alat

  !local variables
  INTEGER :: iat
  INTEGER :: l
  REAL(8), DIMENSION(3,nat) :: xyzred

  CALL cart2frac(nat,alat,rxyz,xyzred)
  DO iat=1,nat
    DO l=1,3
      xyzred(l,iat)=modulo(xyzred(l,iat),1.d0)
    ENDDO
  ENDDO
  CALL frac2cart(nat,alat,xyzred,rxyz)

end subroutine
